/**
 * Shared types for client samples
 *
 * Note: The API returns responses wrapped in {data: ...} format,
 * but the generated SDK types don't reflect this wrapper.
 * These types bridge that gap for type safety.
 */

/**
 * API response wrapper for single resources (create, update, show)
 * The API actually returns {data: T} but OpenAPI spec says it returns T directly
 */
export type ApiDataResponse<T> = {
  data: T;
} | T; // Union type to handle both wrapped and unwrapped responses

/**
 * API response wrapper for list operations (paginated)
 */
export type ApiListResponse<T> = {
  data: T[];
  meta: {
    page: number;
    page_size: number;
    total_count: number;
    total_pages: number;
  };
};

/**
 * Extract data from API response, handling both wrapped and unwrapped formats
 */
export function extractData<T>(response: ApiDataResponse<T>): T {
  if (response && typeof response === 'object' && 'data' in response) {
    return response.data;
  }
  return response as T;
}

/**
 * Extract list data from API response
 */
export function extractListData<T>(response: ApiListResponse<T>): T[] {
  return response.data ?? [];
}
