import { describe, it, expect, beforeAll } from 'vitest';
import { OpenAPI, AuthenticationService } from '@scp/sdk';
import { createTenant } from '@scp/client-samples/tenants/create';
import { createMerchantOrg } from '@scp/client-samples/merchant_orgs/create';
import { createMerchant } from '@scp/client-samples/merchants/create';
import { createAppointmentSearch } from '@scp/client-samples/appointment_searches/create';
import { listAppointmentSearches } from '@scp/client-samples/appointment_searches/list';
import { getAppointmentSearch } from '@scp/client-samples/appointment_searches/show';
import { updateAppointmentSearch } from '@scp/client-samples/appointment_searches/update';
import { deleteAppointmentSearch } from '@scp/client-samples/appointment_searches/delete';


describe('AppointmentSearch API Integration Tests', () => {
  let appointment_searchId: string;
  let testTenantId: string;
  let testMerchantId: string;


  beforeAll(async () => {
    // Authenticate and get token
    const authResponse = await AuthenticationService.scpApiSessionControllerCreate({
      requestBody: {
        email: process.env.ADMIN_EMAIL || 'admin@alvera-scp-dev.local',
        password: process.env.ADMIN_PASSWORD || 'DevPassword123!',
        tenant_name: process.env.TENANT_NAME || 'alvera-scp-dev',
      },
    }) as any;

    OpenAPI.TOKEN = authResponse.token;

    // Create test data using client samples
    const tenant = await createTenant();
    testTenantId = tenant.id;

    const merchantOrg = await createMerchantOrg(testTenantId);
    const merchant = await createMerchant(testTenantId, merchantOrg.id);
    testMerchantId = merchant.id;
  });

  it('should create a appointment_search (201)', async () => {
    const appointment_search = await createAppointmentSearch(testTenantId, testMerchantId);

    expect(appointment_search).toBeDefined();
    expect(appointment_search.id).toBeTruthy();

    appointment_searchId = appointment_search.id;
  });

  it('should list appointmentsearches (200)', async () => {
    const response = await listAppointmentSearches();

    expect(response.data).toBeDefined();
    expect(Array.isArray(response.data)).toBe(true);
    expect(response.meta).toBeDefined();
  });

  it('should get a single appointment_search (200)', async () => {
    const appointment_search = await getAppointmentSearch(appointment_searchId);

    expect(appointment_search).toBeDefined();
    expect(appointment_search.id).toBe(appointment_searchId);
  });

  it('should update a appointment_search (200)', async () => {
    const updated = await updateAppointmentSearch(appointment_searchId, testTenantId, testMerchantId);

    expect(updated).toBeDefined();
    expect(updated.id).toBe(appointment_searchId);
  });

  it('should delete a appointment_search (204)', async () => {
    await deleteAppointmentSearch(appointment_searchId);

    // Verify deletion by trying to get - should throw
    await expect(async () => {
      await getAppointmentSearch(appointment_searchId);
    }).rejects.toThrow();
  });

  // Error case tests
  it('should fail to get non-existent appointment_search (404)', async () => {
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await getAppointmentSearch(fakeId);
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });

  it('should fail to delete non-existent appointment_search (404)', async () => {
    const { AppointmentSearchService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await AppointmentSearchService.appointmentSearchDelete({ id: fakeId });
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });
});
