import { describe, it, expect, beforeAll } from 'vitest';
import { OpenAPI, AuthenticationService } from '@scp/sdk';
import { createTenant } from '@scp/client-samples/tenants/create';
import { createMerchantOrg } from '@scp/client-samples/merchant_orgs/create';
import { createMerchant } from '@scp/client-samples/merchants/create';
import { createEndCustomer } from '@scp/client-samples/endcustomers/create';
import { createService } from '@scp/client-samples/services/create';
import { createBooking } from '@scp/client-samples/bookings/create';
import { createAppointment } from '@scp/client-samples/appointments/create';
import { listAppointments } from '@scp/client-samples/appointments/list';
import { getAppointment } from '@scp/client-samples/appointments/show';
import { updateAppointment } from '@scp/client-samples/appointments/update';
import { deleteAppointment } from '@scp/client-samples/appointments/delete';

describe('Appointment API Integration Tests', () => {
  let appointmentId: string;
  let testTenantId: string;
  let testMerchantId: string;
  let testEndCustomerId: string;
  let testServiceId: string;
  let testBookingId: string;

  beforeAll(async () => {
    // Authenticate and get token
    const authResponse = await AuthenticationService.scpApiSessionControllerCreate({
      requestBody: {
        email: process.env.ADMIN_EMAIL || 'admin@alvera-scp-dev.local',
        password: process.env.ADMIN_PASSWORD || 'DevPassword123!',
        tenant_name: process.env.TENANT_NAME || 'alvera-scp-dev',
      },
    }) as any;

    OpenAPI.TOKEN = authResponse.token;

    // Create test data using client samples
    const tenant = await createTenant();
    testTenantId = tenant.id;

    const merchantOrg = await createMerchantOrg(testTenantId);
    const merchant = await createMerchant(testTenantId, merchantOrg.id);
    testMerchantId = merchant.id;

    const endCustomer = await createEndCustomer(testTenantId, testMerchantId);
    testEndCustomerId = endCustomer.id;

    const service = await createService(testTenantId, testMerchantId);
    testServiceId = service.id;

    const booking = await createBooking(testTenantId, testMerchantId, testEndCustomerId);
    testBookingId = booking.id;
  });

  it('should create an appointment (201)', async () => {
    const appointment = await createAppointment(
      testTenantId,
      testMerchantId,
      testEndCustomerId,
      testServiceId,
      testBookingId
    );

    expect(appointment).toBeDefined();
    expect(appointment.id).toBeTruthy();
    expect(appointment.status).toBe('pending');
    expect(appointment.starts_at).toBeTruthy();
    expect(appointment.ends_at).toBeTruthy();

    appointmentId = appointment.id;
  });

  it('should list appointments (200)', async () => {
    const response = await listAppointments();

    expect(response.data).toBeDefined();
    expect(Array.isArray(response.data)).toBe(true);
    expect(response.data!.length).toBeGreaterThan(0);
    expect(response.meta).toBeDefined();
    expect(response.meta!.total_count).toBeGreaterThan(0);

    const foundAppointment = response.data!.find((a) => a.id === appointmentId);
    expect(foundAppointment).toBeDefined();
  });

  it('should get a single appointment (200)', async () => {
    const appointment = await getAppointment(appointmentId);

    expect(appointment).toBeDefined();
    expect(appointment.id).toBe(appointmentId);
    expect(appointment.status).toBe('pending');
  });

  it('should update an appointment (200)', async () => {
    const updated = await updateAppointment(
      appointmentId,
      testTenantId,
      testMerchantId,
      testEndCustomerId,
      testServiceId,
      testBookingId
    );

    expect(updated).toBeDefined();
    expect(updated.id).toBe(appointmentId);
    expect(updated.status).toBe('confirmed');
  });

  it('should delete an appointment (200)', async () => {
    await deleteAppointment(appointmentId);

    // Verify deletion by trying to get - should throw
    await expect(async () => {
      await getAppointment(appointmentId);
    }).rejects.toThrow();
  });

  // Error case tests
  it('should fail to create appointment with missing required fields (422)', async () => {
    const { AppointmentService } = await import('@scp/sdk');

    try {
      await AppointmentService.appointmentCreate({
        requestBody: {
          // Missing required fields
          notes: 'Invalid appointment',
        } as any,
      });
      expect.fail('Should have thrown validation error');
    } catch (error: any) {
      expect(error.status).toBe(422);
      expect(error.body).toBeDefined();
    }
  });

  it('should fail to create appointment with invalid status enum (422)', async () => {
    const { AppointmentService } = await import('@scp/sdk');

    try {
      await AppointmentService.appointmentCreate({
        requestBody: {
          status: 'invalid_status',
          starts_at: new Date().toISOString(),
          ends_at: new Date(Date.now() + 3600000).toISOString(),
          tenant_id: testTenantId,
          merchant_id: testMerchantId,
          end_customer_id: testEndCustomerId,
          service_id: testServiceId,
          booking_id: testBookingId,
        } as any,
      });
      expect.fail('Should have thrown validation error');
    } catch (error: any) {
      expect(error.status).toBe(422);
      expect(error.body).toBeDefined();
    }
  });

  it('should fail to get non-existent appointment (404)', async () => {
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await getAppointment(fakeId);
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });

  it('should fail to update non-existent appointment (404 or 422)', async () => {
    const { AppointmentService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await AppointmentService.appointmentUpdate({
        id: fakeId,
        requestBody: {
          status: 'confirmed',
          starts_at: new Date().toISOString(),
          ends_at: new Date(Date.now() + 3600000).toISOString(),
          tenant_id: testTenantId,
          merchant_id: testMerchantId,
          end_customer_id: testEndCustomerId,
          service_id: testServiceId,
          booking_id: testBookingId,
        } as any,
      });
      expect.fail('Should have thrown error');
    } catch (error: any) {
      if (error.status) {
        expect([404, 422]).toContain(error.status);
      } else {
        // Any error is acceptable for non-existent resource
        expect(error).toBeDefined();
      }
    }
  });

  it('should fail to delete non-existent appointment (404)', async () => {
    const { AppointmentService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await AppointmentService.appointmentDelete({ id: fakeId });
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      if (error.status) {
        expect(error.status).toBe(404);
      } else {
        // Any error is acceptable for non-existent resource
        expect(error).toBeDefined();
      }
    }
  });
});
