import { describe, it, expect, beforeAll } from 'vitest';
import { OpenAPI, AuthenticationService } from '@scp/sdk';
import { createTenant } from '@scp/client-samples/tenants/create';
import { createMerchantOrg } from '@scp/client-samples/merchant_orgs/create';
import { createMerchant } from '@scp/client-samples/merchants/create';
import { createService } from '@scp/client-samples/services/create';
import { createProvider } from '@scp/client-samples/providers/create';
import { createLocation } from '@scp/client-samples/locations/create';
import { createAvailability } from '@scp/client-samples/availabilities/create';
import { listAvailabilities } from '@scp/client-samples/availabilities/list';
import { getAvailability } from '@scp/client-samples/availabilities/show';
import { updateAvailability } from '@scp/client-samples/availabilities/update';
import { deleteAvailability } from '@scp/client-samples/availabilities/delete';

describe('Availability API Integration Tests', () => {
  let availabilityId: string;
  let testTenantId: string;
  let testMerchantId: string;
  let testServiceId: string;
  let testProviderId: string;
  let testLocationId: string;

  beforeAll(async () => {
    // Authenticate and get token
    const authResponse = await AuthenticationService.scpApiSessionControllerCreate({
      requestBody: {
        email: process.env.ADMIN_EMAIL || 'admin@alvera-scp-dev.local',
        password: process.env.ADMIN_PASSWORD || 'DevPassword123!',
        tenant_name: process.env.TENANT_NAME || 'alvera-scp-dev',
      },
    }) as any;

    OpenAPI.TOKEN = authResponse.token;

    // Create test data using client samples
    const tenant = await createTenant();
    testTenantId = tenant.id;

    const merchantOrg = await createMerchantOrg(testTenantId);
    const merchant = await createMerchant(testTenantId, merchantOrg.id);
    testMerchantId = merchant.id;

    const service = await createService(testTenantId, testMerchantId);
    testServiceId = service.id;

    const provider = await createProvider(testTenantId, testMerchantId);
    testProviderId = provider.id;

    const location = await createLocation(testTenantId, testMerchantId);
    testLocationId = location.id;
  });

  it('should create an availability (201)', async () => {
    const availability = await createAvailability(
      testTenantId,
      testMerchantId,
      testServiceId,
      testProviderId,
      testLocationId
    );

    expect(availability).toBeDefined();
    expect(availability.id).toBeTruthy();
    expect(availability.status).toBe('open');
    expect(availability.start_at).toBeTruthy();
    expect(availability.duration_minutes).toBe(60);

    availabilityId = availability.id;
  });

  it('should list availabilities (200)', async () => {
    const response = await listAvailabilities();

    expect(response.data).toBeDefined();
    expect(Array.isArray(response.data)).toBe(true);
    expect(response.data!.length).toBeGreaterThan(0);
    expect(response.meta).toBeDefined();
    expect(response.meta!.total_count).toBeGreaterThan(0);

    const foundAvailability = response.data!.find((a) => a.id === availabilityId);
    expect(foundAvailability).toBeDefined();
  });

  it('should get a single availability (200)', async () => {
    const availability = await getAvailability(availabilityId);

    expect(availability).toBeDefined();
    expect(availability.id).toBe(availabilityId);
    expect(availability.status).toBe('open');
  });

  it('should update an availability (200)', async () => {
    const updated = await updateAvailability(
      availabilityId,
      testTenantId,
      testMerchantId,
      testServiceId,
      testProviderId,
      testLocationId
    );

    expect(updated).toBeDefined();
    expect(updated.id).toBe(availabilityId);
    expect(updated.status).toBe('booked');
    expect(updated.duration_minutes).toBe(90);
  });

  it('should delete an availability (200)', async () => {
    await deleteAvailability(availabilityId);

    // Verify deletion by trying to get - should throw
    await expect(async () => {
      await getAvailability(availabilityId);
    }).rejects.toThrow();
  });

  // Error case tests
  it('should fail to create availability with missing required fields (422)', async () => {
    const { AvailabilityService } = await import('@scp/sdk');

    try {
      await AvailabilityService.availabilityCreate({
        requestBody: {
          // Missing required fields
          status: 'open',
        } as any,
      });
      expect.fail('Should have thrown validation error');
    } catch (error: any) {
      expect(error.status).toBe(422);
      expect(error.body).toBeDefined();
    }
  });

  it('should fail to create availability with invalid status enum (422)', async () => {
    const { AvailabilityService } = await import('@scp/sdk');

    try {
      await AvailabilityService.availabilityCreate({
        requestBody: {
          status: 'invalid_status',
          start_at: new Date().toISOString(),
          duration_minutes: 60,
          tenant_id: testTenantId,
          merchant_id: testMerchantId,
          service_id: testServiceId,
        } as any,
      });
      expect.fail('Should have thrown validation error');
    } catch (error: any) {
      expect(error.status).toBe(422);
      expect(error.body).toBeDefined();
    }
  });

  it('should fail to get non-existent availability (404)', async () => {
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await getAvailability(fakeId);
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });

  it('should fail to update non-existent availability (404 or 422)', async () => {
    const { AvailabilityService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await AvailabilityService.availabilityUpdate({
        id: fakeId,
        requestBody: {
          status: 'booked',
          start_at: new Date().toISOString(),
          duration_minutes: 60,
          tenant_id: testTenantId,
          merchant_id: testMerchantId,
          service_id: testServiceId,
        } as any,
      });
      expect.fail('Should have thrown error');
    } catch (error: any) {
      if (error.status) {
        expect([404, 422]).toContain(error.status);
      } else {
        // Fallback for errors without status
        expect(error).toBeDefined();
      }
    }
  });

  it('should fail to delete non-existent availability (404)', async () => {
    const { AvailabilityService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await AvailabilityService.availabilityDelete({ id: fakeId });
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      if (error.status) {
        expect(error.status).toBe(404);
      } else {
        // Fallback for errors without status
        expect(error).toBeDefined();
      }
    }
  });
});
