import { describe, it, expect, beforeAll } from 'vitest';
import { OpenAPI, AuthenticationService } from '@scp/sdk';
import { createTenant } from '@scp/client-samples/tenants/create';
import { createMerchantOrg } from '@scp/client-samples/merchant_orgs/create';
import { createMerchant } from '@scp/client-samples/merchants/create';
import { createEndCustomer } from '@scp/client-samples/endcustomers/create';
import { createService } from '@scp/client-samples/services/create';
import { createBookingEnquiry } from '@scp/client-samples/booking_enquiries/create';
import { listBookingEnquiries } from '@scp/client-samples/booking_enquiries/list';
import { getBookingEnquiry } from '@scp/client-samples/booking_enquiries/show';
import { updateBookingEnquiry } from '@scp/client-samples/booking_enquiries/update';
import { deleteBookingEnquiry } from '@scp/client-samples/booking_enquiries/delete';

describe('BookingEnquiry API Integration Tests', () => {
  let bookingEnquiryId: string;
  let testTenantId: string;
  let testMerchantId: string;
  let testEndCustomerId: string;
  let testServiceIds: string[];

  beforeAll(async () => {
    // Authenticate and get token
    const authResponse = await AuthenticationService.scpApiSessionControllerCreate({
      requestBody: {
        email: process.env.ADMIN_EMAIL || 'admin@alvera-scp-dev.local',
        password: process.env.ADMIN_PASSWORD || 'DevPassword123!',
        tenant_name: process.env.TENANT_NAME || 'alvera-scp-dev',
      },
    }) as any;

    OpenAPI.TOKEN = authResponse.token;

    // Create test data using client samples
    const tenant = await createTenant();
    testTenantId = tenant.id;

    const merchantOrg = await createMerchantOrg(testTenantId);
    const merchant = await createMerchant(testTenantId, merchantOrg.id);
    testMerchantId = merchant.id;

    const endCustomer = await createEndCustomer(testTenantId, testMerchantId);
    testEndCustomerId = endCustomer.id;

    // Create a service for the booking enquiry
    const service = await createService(testTenantId, testMerchantId);
    testServiceIds = [service.id];
  });

  it('should create a booking_enquiry (201)', async () => {
    const bookingEnquiry = await createBookingEnquiry(
      testTenantId,
      testMerchantId,
      testEndCustomerId,
      testServiceIds
    );

    expect(bookingEnquiry).toBeDefined();
    expect(bookingEnquiry.id).toBeTruthy();
    expect(bookingEnquiry.status).toBe('searched');

    bookingEnquiryId = bookingEnquiry.id;
  });

  it('should list booking_enquiries (200)', async () => {
    const response = await listBookingEnquiries();

    expect(response.data).toBeDefined();
    expect(Array.isArray(response.data)).toBe(true);
    expect(response.data!.length).toBeGreaterThan(0);
    expect(response.meta).toBeDefined();
    expect(response.meta!.total_count).toBeGreaterThan(0);

    const foundBookingEnquiry = response.data!.find((be) => be.id === bookingEnquiryId);
    expect(foundBookingEnquiry).toBeDefined();
  });

  it('should get a single booking_enquiry (200)', async () => {
    const bookingEnquiry = await getBookingEnquiry(bookingEnquiryId);

    expect(bookingEnquiry).toBeDefined();
    expect(bookingEnquiry.id).toBe(bookingEnquiryId);
    expect(bookingEnquiry.status).toBe('searched');
  });

  it('should update a booking_enquiry (200)', async () => {
    const updated = await updateBookingEnquiry(
      bookingEnquiryId,
      testTenantId,
      testMerchantId,
      testEndCustomerId,
      testServiceIds
    );

    expect(updated).toBeDefined();
    expect(updated.id).toBe(bookingEnquiryId);
    expect(updated.status).toBe('searched');
  });

  it('should delete a booking_enquiry (204)', async () => {
    await deleteBookingEnquiry(bookingEnquiryId);

    // Verify deletion by trying to get - should throw
    await expect(async () => {
      await getBookingEnquiry(bookingEnquiryId);
    }).rejects.toThrow();
  });

  // Error case tests
  it('should fail to create booking_enquiry with missing service_ids (422)', async () => {
    const { BookingEnquiryService } = await import('@scp/sdk');

    try {
      await BookingEnquiryService.bookingEnquiryCreate({
        requestBody: {
          status: 'searched',
          tenant_id: testTenantId,
          merchant_id: testMerchantId,
          end_customer_id: testEndCustomerId,
          service_ids: [], // Empty - should fail validation
        } as any,
      });
      expect.fail('Should have thrown validation error');
    } catch (error: any) {
      if (error.status) {
        expect(error.status).toBe(422);
      } else {
        expect(error).toBeDefined();
      }
    }
  });

  it('should fail to get non-existent booking_enquiry (404)', async () => {
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await getBookingEnquiry(fakeId);
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      if (error.status) {
        expect(error.status).toBe(404);
      } else {
        expect(error).toBeDefined();
      }
    }
  });

  it('should fail to update non-existent booking_enquiry (404)', async () => {
    const { BookingEnquiryService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await BookingEnquiryService.bookingEnquiryUpdate({
        id: fakeId,
        requestBody: {
          status: 'searched',
          tenant_id: testTenantId,
          merchant_id: testMerchantId,
          end_customer_id: testEndCustomerId,
          service_ids: testServiceIds,
        } as any,
      });
      expect.fail('Should have thrown error');
    } catch (error: any) {
      if (error.status) {
        expect([404, 422]).toContain(error.status);
      } else {
        expect(error).toBeDefined();
      }
    }
  });

  it('should fail to delete non-existent booking_enquiry (404)', async () => {
    const { BookingEnquiryService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await BookingEnquiryService.bookingEnquiryDelete({ id: fakeId });
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      if (error.status) {
        expect(error.status).toBe(404);
      } else {
        expect(error).toBeDefined();
      }
    }
  });
});
