import { describe, it, expect, beforeAll } from 'vitest';
import { OpenAPI, AuthenticationService } from '@scp/sdk';
import { createTenant } from '@scp/client-samples/tenants/create';
import { createMerchantOrg } from '@scp/client-samples/merchant_orgs/create';
import { createMerchant } from '@scp/client-samples/merchants/create';
import { createBookingPolicy } from '@scp/client-samples/booking_policies/create';
import { listBookingPolicies } from '@scp/client-samples/booking_policies/list';
import { getBookingPolicy } from '@scp/client-samples/booking_policies/show';
import { updateBookingPolicy } from '@scp/client-samples/booking_policies/update';
import { deleteBookingPolicy } from '@scp/client-samples/booking_policies/delete';


describe('BookingPolicy API Integration Tests', () => {
  let booking_policyId: string;
  let testTenantId: string;
  let testMerchantId: string;


  beforeAll(async () => {
    // Authenticate and get token
    const authResponse = await AuthenticationService.scpApiSessionControllerCreate({
      requestBody: {
        email: process.env.ADMIN_EMAIL || 'admin@alvera-scp-dev.local',
        password: process.env.ADMIN_PASSWORD || 'DevPassword123!',
        tenant_name: process.env.TENANT_NAME || 'alvera-scp-dev',
      },
    }) as any;

    OpenAPI.TOKEN = authResponse.token;

    // Create test data using client samples
    const tenant = await createTenant();
    testTenantId = tenant.id;

    const merchantOrg = await createMerchantOrg(testTenantId);
    const merchant = await createMerchant(testTenantId, merchantOrg.id);
    testMerchantId = merchant.id;
  });

  it('should create a booking_policy (201)', async () => {
    const booking_policy = await createBookingPolicy(testTenantId, testMerchantId);

    expect(booking_policy).toBeDefined();
    expect(booking_policy.id).toBeTruthy();

    booking_policyId = booking_policy.id;
  });

  it('should list bookingpolicies (200)', async () => {
    const response = await listBookingPolicies();

    expect(response.data).toBeDefined();
    expect(Array.isArray(response.data)).toBe(true);
    expect(response.meta).toBeDefined();
  });

  it('should get a single booking_policy (200)', async () => {
    const booking_policy = await getBookingPolicy(booking_policyId);

    expect(booking_policy).toBeDefined();
    expect(booking_policy.id).toBe(booking_policyId);
  });

  it('should update a booking_policy (200)', async () => {
    const updated = await updateBookingPolicy(booking_policyId, testTenantId, testMerchantId);

    expect(updated).toBeDefined();
    expect(updated.id).toBe(booking_policyId);
  });

  it('should delete a booking_policy (204)', async () => {
    await deleteBookingPolicy(booking_policyId);

    // Verify deletion by trying to get - should throw
    await expect(async () => {
      await getBookingPolicy(booking_policyId);
    }).rejects.toThrow();
  });

  // Error case tests
  it('should fail to get non-existent booking_policy (404)', async () => {
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await getBookingPolicy(fakeId);
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });

  it('should fail to delete non-existent booking_policy (404)', async () => {
    const { BookingPolicyService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await BookingPolicyService.bookingPolicyDelete({ id: fakeId });
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });
});
