import { describe, it, expect, beforeAll } from 'vitest';
import { OpenAPI, AuthenticationService } from '@scp/sdk';
import { createTenant } from '@scp/client-samples/tenants/create';
import { createMerchantOrg } from '@scp/client-samples/merchant_orgs/create';
import { createMerchant } from '@scp/client-samples/merchants/create';
import { createEndCustomer } from '@scp/client-samples/endcustomers/create';
import { createFormResponse } from '@scp/client-samples/form_responses/create';
import { listFormResponses } from '@scp/client-samples/form_responses/list';
import { getFormResponse } from '@scp/client-samples/form_responses/show';
import { updateFormResponse } from '@scp/client-samples/form_responses/update';
import { deleteFormResponse } from '@scp/client-samples/form_responses/delete';

describe('FormResponse API Integration Tests', () => {
  let formResponseId: string;
  let testTenantId: string;
  let testMerchantOrgId: string;
  let testMerchantId: string;
  let testEndCustomerId: string;

  beforeAll(async () => {
    // Authenticate and get token
    const authResponse = await AuthenticationService.scpApiSessionControllerCreate({
      requestBody: {
        email: process.env.ADMIN_EMAIL || 'admin@alvera-scp-dev.local',
        password: process.env.ADMIN_PASSWORD || 'DevPassword123!',
        tenant_name: process.env.TENANT_NAME || 'alvera-scp-dev',
      },
    }) as any;

    OpenAPI.TOKEN = authResponse.token;

    // Create test data using client samples
    const tenant = await createTenant();
    testTenantId = tenant.id;

    const merchantOrg = await createMerchantOrg(testTenantId);
    testMerchantOrgId = merchantOrg.id;

    const merchant = await createMerchant(testTenantId, testMerchantOrgId);
    testMerchantId = merchant.id;

    const endCustomer = await createEndCustomer(testTenantId, testMerchantId);
    testEndCustomerId = endCustomer.id;
  });

  it('should create a form response (201)', async () => {
    const formResponse = await createFormResponse(testTenantId, testMerchantId, testEndCustomerId);

    expect(formResponse).toBeDefined();
    expect(formResponse.id).toBeTruthy();
    expect(formResponse.name).toBeTruthy();
    expect(formResponse.form_type).toBeTruthy();
    expect(formResponse.tenant_id).toBe(testTenantId);
    expect(formResponse.merchant_id).toBe(testMerchantId);
    expect(formResponse.end_customer_id).toBe(testEndCustomerId);

    formResponseId = formResponse.id;
  });

  it('should list form responses (200)', async () => {
    const response = await listFormResponses();

    expect(response.data).toBeDefined();
    expect(Array.isArray(response.data)).toBe(true);
    expect(response.data!.length).toBeGreaterThan(0);
    expect(response.meta).toBeDefined();
    expect(response.meta!.total_count).toBeGreaterThan(0);

    const found = response.data!.find((fr) => fr.id === formResponseId);
    expect(found).toBeDefined();
  });

  it('should get a single form response (200)', async () => {
    const formResponse = await getFormResponse(formResponseId);

    expect(formResponse).toBeDefined();
    expect(formResponse.id).toBe(formResponseId);
    expect(formResponse.name).toBeTruthy();
    expect(formResponse.form_type).toBeTruthy();
  });

  it('should update a form response (200)', async () => {
    const updated = await updateFormResponse(formResponseId, testTenantId, testMerchantId, testEndCustomerId);

    expect(updated).toBeDefined();
    expect(updated.id).toBe(formResponseId);
    expect(updated.name).toBeTruthy();
  });

  it('should delete a form response (204)', async () => {
    await deleteFormResponse(formResponseId);

    // Verify deletion by trying to get - should throw
    await expect(async () => {
      await getFormResponse(formResponseId);
    }).rejects.toThrow();
  });

  // Error case tests
  it('should fail to create form response with missing required fields (422)', async () => {
    const { FormResponseService } = await import('@scp/sdk');

    try {
      await FormResponseService.formResponseCreate({
        requestBody: {
          // Missing required fields: name, form_type, submission_date, etc.
          tenant_id: testTenantId,
        } as any,
      });
      expect.fail('Should have thrown validation error');
    } catch (error: any) {
      expect(error.status).toBe(422);
      expect(error.body).toBeDefined();
    }
  });

  it('should fail to get non-existent form response (404)', async () => {
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await getFormResponse(fakeId);
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });

  it('should fail to update non-existent form response (404)', async () => {
    const { FormResponseService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await FormResponseService.formResponseUpdate({
        id: fakeId,
        requestBody: {
          name: 'Updated Form',
          form_type: 'intake',
          submission_date: new Date().toISOString(),
          tenant_id: testTenantId,
          merchant_id: testMerchantId,
          end_customer_id: testEndCustomerId,
        } as any,
      });
      expect.fail('Should have thrown error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });

  it('should fail to delete non-existent form response (404)', async () => {
    const { FormResponseService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await FormResponseService.formResponseDelete({ id: fakeId });
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });
});
