import { describe, it, expect, beforeAll } from 'vitest';
import { OpenAPI, AuthenticationService, MessageService } from '@scp/sdk';
import { createTenant } from '@scp/client-samples/tenants/create';
import { createMerchantOrg } from '@scp/client-samples/merchant_orgs/create';
import { createMerchant } from '@scp/client-samples/merchants/create';
import { createEndCustomer } from '@scp/client-samples/endcustomers/create';
import { createMessage } from '@scp/client-samples/messages/create';
import { listMessages } from '@scp/client-samples/messages/list';
import { getMessage } from '@scp/client-samples/messages/show';
import { updateMessage } from '@scp/client-samples/messages/update';
import { deleteMessage } from '@scp/client-samples/messages/delete';

describe('Message API Integration Tests', () => {
  let messageId: string;
  let testTenantId: string;
  let testMerchantId: string;
  let testEndCustomerId: string;
  // Using a dummy plugin ID since plugin creation isn't exposed via API
  const testPluginId = '00000000-0000-0000-0000-000000000001';

  beforeAll(async () => {
    // Authenticate
    const authResponse = (await AuthenticationService.scpApiSessionControllerCreate({
      requestBody: {
        email: process.env.ADMIN_EMAIL || 'admin@alvera-scp-dev.local',
        password: process.env.ADMIN_PASSWORD || 'DevPassword123!',
        tenant_name: process.env.TENANT_NAME || 'alvera-scp-dev',
      },
    })) as any;

    OpenAPI.TOKEN = authResponse.token;

    // Setup test data
    const tenant = await createTenant();
    testTenantId = tenant.id;

    const merchantOrg = await createMerchantOrg(testTenantId);
    const merchant = await createMerchant(testTenantId, merchantOrg.id);
    testMerchantId = merchant.id;

    const endCustomer = await createEndCustomer(testTenantId, testMerchantId);
    testEndCustomerId = endCustomer.id;
  });

  it('should create a message (201)', async () => {
    const message = await createMessage(
      testTenantId,
      testMerchantId,
      testEndCustomerId,
      testPluginId
    );

    expect(message).toBeDefined();
    expect(message.id).toBeTruthy();
    expect(message.body).toBeTruthy();

    messageId = message.id;
  });

  it('should list messages (200)', async () => {
    const response = await listMessages();

    expect(response.data).toBeDefined();
    expect(Array.isArray(response.data)).toBe(true);
  });

  it('should get a single message (200)', async () => {
    const message = await getMessage(messageId);

    expect(message.id).toBe(messageId);
  });

  it('should update a message (200)', async () => {
    const updated = await updateMessage(
      messageId,
      testTenantId,
      testMerchantId,
      testEndCustomerId,
      testPluginId
    );

    expect(updated.id).toBe(messageId);
  });

  it('should delete a message (204)', async () => {
    await deleteMessage(messageId);
  });

  // Error cases
  it('should fail to create message with missing fields (422)', async () => {
    try {
      await MessageService.messageCreate({
        requestBody: { body: 'test' } as any,
      });
      expect.fail('Should have thrown validation error');
    } catch (error: any) {
      expect(error.status).toBe(422);
    }
  });

  it('should fail to get non-existent message (404)', async () => {
    const fakeId = '00000000-0000-0000-0000-000000000000';
    try {
      await getMessage(fakeId);
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });
});
