import { describe, it, expect, beforeAll } from 'vitest';
import { OpenAPI, AuthenticationService } from '@scp/sdk';
import { createTenant } from '@scp/client-samples/tenants/create';
import { createMerchantOrg } from '@scp/client-samples/merchant_orgs/create';
import { createMerchant } from '@scp/client-samples/merchants/create';
import { createNotificationPolicy } from '@scp/client-samples/notification_policies/create';
import { listNotificationPolicies } from '@scp/client-samples/notification_policies/list';
import { getNotificationPolicy } from '@scp/client-samples/notification_policies/show';
import { updateNotificationPolicy } from '@scp/client-samples/notification_policies/update';
import { deleteNotificationPolicy } from '@scp/client-samples/notification_policies/delete';


describe('NotificationPolicy API Integration Tests', () => {
  let notification_policyId: string;
  let testTenantId: string;
  let testMerchantId: string;


  beforeAll(async () => {
    // Authenticate and get token
    const authResponse = await AuthenticationService.scpApiSessionControllerCreate({
      requestBody: {
        email: process.env.ADMIN_EMAIL || 'admin@alvera-scp-dev.local',
        password: process.env.ADMIN_PASSWORD || 'DevPassword123!',
        tenant_name: process.env.TENANT_NAME || 'alvera-scp-dev',
      },
    }) as any;

    OpenAPI.TOKEN = authResponse.token;

    // Create test data using client samples
    const tenant = await createTenant();
    testTenantId = tenant.id;

    const merchantOrg = await createMerchantOrg(testTenantId);
    const merchant = await createMerchant(testTenantId, merchantOrg.id);
    testMerchantId = merchant.id;
  });

  it('should create a notification_policy (201)', async () => {
    const notification_policy = await createNotificationPolicy(testTenantId, testMerchantId);

    expect(notification_policy).toBeDefined();
    expect(notification_policy.id).toBeTruthy();

    notification_policyId = notification_policy.id;
  });

  it('should list notificationpolicies (200)', async () => {
    const response = await listNotificationPolicies();

    expect(response.data).toBeDefined();
    expect(Array.isArray(response.data)).toBe(true);
    expect(response.meta).toBeDefined();
  });

  it('should get a single notification_policy (200)', async () => {
    const notification_policy = await getNotificationPolicy(notification_policyId);

    expect(notification_policy).toBeDefined();
    expect(notification_policy.id).toBe(notification_policyId);
  });

  it('should update a notification_policy (200)', async () => {
    const updated = await updateNotificationPolicy(notification_policyId, testTenantId, testMerchantId);

    expect(updated).toBeDefined();
    expect(updated.id).toBe(notification_policyId);
  });

  it('should delete a notification_policy (204)', async () => {
    await deleteNotificationPolicy(notification_policyId);

    // Verify deletion by trying to get - should throw
    await expect(async () => {
      await getNotificationPolicy(notification_policyId);
    }).rejects.toThrow();
  });

  // Error case tests
  it('should fail to get non-existent notification_policy (404)', async () => {
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await getNotificationPolicy(fakeId);
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });

  it('should fail to delete non-existent notification_policy (404)', async () => {
    const { NotificationPolicyService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await NotificationPolicyService.notificationPolicyDelete({ id: fakeId });
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });
});
