import { describe, it, expect, beforeAll } from 'vitest';
import { OpenAPI, AuthenticationService } from '@scp/sdk';
import { createTenant } from '@scp/client-samples/tenants/create';
import { createMerchantOrg } from '@scp/client-samples/merchant_orgs/create';
import { createMerchant } from '@scp/client-samples/merchants/create';
import { createEndCustomer } from '@scp/client-samples/endcustomers/create';
import { createBooking } from '@scp/client-samples/bookings/create';
import { createPaymentContract } from '@scp/client-samples/payment_contracts/create';
import { listPaymentContracts } from '@scp/client-samples/payment_contracts/list';
import { getPaymentContract } from '@scp/client-samples/payment_contracts/show';
import { updatePaymentContract } from '@scp/client-samples/payment_contracts/update';
import { deletePaymentContract } from '@scp/client-samples/payment_contracts/delete';


describe('PaymentContract API Integration Tests', () => {
  let payment_contractId: string;
  let testTenantId: string;
  let testMerchantId: string;
  let testBookingId: string;
  let testEndCustomerId: string;


  beforeAll(async () => {
    // Authenticate and get token
    const authResponse = await AuthenticationService.scpApiSessionControllerCreate({
      requestBody: {
        email: process.env.ADMIN_EMAIL || 'admin@alvera-scp-dev.local',
        password: process.env.ADMIN_PASSWORD || 'DevPassword123!',
        tenant_name: process.env.TENANT_NAME || 'alvera-scp-dev',
      },
    }) as any;

    OpenAPI.TOKEN = authResponse.token;

    // Create test data using client samples
    const tenant = await createTenant();
    testTenantId = tenant.id;

    const merchantOrg = await createMerchantOrg(testTenantId);
    const merchant = await createMerchant(testTenantId, merchantOrg.id);
    testMerchantId = merchant.id;

    const endCustomer = await createEndCustomer(testTenantId, merchant.id);
    testEndCustomerId = endCustomer.id;

    const booking = await createBooking(testTenantId, merchant.id, endCustomer.id);
    testBookingId = booking.id;
  });

  it('should create a payment_contract (201)', async () => {
    const payment_contract = await createPaymentContract(testTenantId, testMerchantId, testBookingId, testEndCustomerId);

    expect(payment_contract).toBeDefined();
    expect(payment_contract.id).toBeTruthy();

    payment_contractId = payment_contract.id;
  });

  it('should list paymentcontracts (200)', async () => {
    const response = await listPaymentContracts();

    expect(response.data).toBeDefined();
    expect(Array.isArray(response.data)).toBe(true);
    expect(response.meta).toBeDefined();
  });

  it('should get a single payment_contract (200)', async () => {
    const payment_contract = await getPaymentContract(payment_contractId);

    expect(payment_contract).toBeDefined();
    expect(payment_contract.id).toBe(payment_contractId);
  });

  it('should update a payment_contract (200)', async () => {
    const updated = await updatePaymentContract(payment_contractId, testTenantId, testMerchantId, testBookingId, testEndCustomerId);

    expect(updated).toBeDefined();
    expect(updated.id).toBe(payment_contractId);
  });

  it('should delete a payment_contract (204)', async () => {
    await deletePaymentContract(payment_contractId);

    // Verify deletion by trying to get - should throw
    await expect(async () => {
      await getPaymentContract(payment_contractId);
    }).rejects.toThrow();
  });

  // Error case tests
  it('should fail to get non-existent payment_contract (404)', async () => {
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await getPaymentContract(fakeId);
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });

  it('should fail to delete non-existent payment_contract (404)', async () => {
    const { PaymentContractService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await PaymentContractService.paymentContractDelete({ id: fakeId });
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });
});
