import { describe, it, expect, beforeAll } from 'vitest';
import { OpenAPI, AuthenticationService } from '@scp/sdk';
import { createTenant } from '@scp/client-samples/tenants/create';
import { createMerchantOrg } from '@scp/client-samples/merchant_orgs/create';
import { createMerchant } from '@scp/client-samples/merchants/create';
import { createPaymentPolicy } from '@scp/client-samples/payment_policies/create';
import { listPaymentPolicies } from '@scp/client-samples/payment_policies/list';
import { getPaymentPolicy } from '@scp/client-samples/payment_policies/show';
import { updatePaymentPolicy } from '@scp/client-samples/payment_policies/update';
import { deletePaymentPolicy } from '@scp/client-samples/payment_policies/delete';


describe('PaymentPolicy API Integration Tests', () => {
  let payment_policyId: string;
  let testTenantId: string;
  let testMerchantOrgId: string;
  let testMerchantId: string;

  beforeAll(async () => {
    // Authenticate and get token
    const authResponse = await AuthenticationService.scpApiSessionControllerCreate({
      requestBody: {
        email: process.env.ADMIN_EMAIL || 'admin@alvera-scp-dev.local',
        password: process.env.ADMIN_PASSWORD || 'DevPassword123!',
        tenant_name: process.env.TENANT_NAME || 'alvera-scp-dev',
      },
    }) as any;

    OpenAPI.TOKEN = authResponse.token;

    // Create test data using client samples
    const tenant = await createTenant();
    testTenantId = tenant.id;

    const merchantOrg = await createMerchantOrg(testTenantId);
    testMerchantOrgId = merchantOrg.id;

    const merchant = await createMerchant(testTenantId, merchantOrg.id);
    testMerchantId = merchant.id;
  });

  it('should create a payment_policy (201)', async () => {
    const payment_policy = await createPaymentPolicy(testTenantId, testMerchantOrgId, testMerchantId);

    expect(payment_policy).toBeDefined();
    expect(payment_policy.id).toBeTruthy();

    payment_policyId = payment_policy.id;
  });

  it('should list paymentpolicies (200)', async () => {
    const response = await listPaymentPolicies();

    expect(response.data).toBeDefined();
    expect(Array.isArray(response.data)).toBe(true);
    expect(response.meta).toBeDefined();
  });

  it('should get a single payment_policy (200)', async () => {
    const payment_policy = await getPaymentPolicy(payment_policyId);

    expect(payment_policy).toBeDefined();
    expect(payment_policy.id).toBe(payment_policyId);
  });

  it('should update a payment_policy (200)', async () => {
    const updated = await updatePaymentPolicy(payment_policyId, testTenantId, testMerchantOrgId, testMerchantId);

    expect(updated).toBeDefined();
    expect(updated.id).toBe(payment_policyId);
  });

  it('should delete a payment_policy (204)', async () => {
    await deletePaymentPolicy(payment_policyId);

    // Verify deletion by trying to get - should throw
    await expect(async () => {
      await getPaymentPolicy(payment_policyId);
    }).rejects.toThrow();
  });

  // Error case tests
  it('should fail to get non-existent payment_policy (404)', async () => {
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await getPaymentPolicy(fakeId);
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });

  it('should fail to delete non-existent payment_policy (404)', async () => {
    const { PaymentPolicyService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await PaymentPolicyService.paymentPolicyDelete({ id: fakeId });
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });
});
