import { describe, it, expect, beforeAll } from 'vitest';
import { OpenAPI, AuthenticationService } from '@scp/sdk';
import { createTenant } from '@scp/client-samples/tenants/create';
import { listTenants } from '@scp/client-samples/tenants/list';
import { getTenant } from '@scp/client-samples/tenants/show';
import { updateTenant } from '@scp/client-samples/tenants/update';
import { deleteTenant } from '@scp/client-samples/tenants/delete';

describe('Tenant API Integration Tests', () => {
  let tenantId: string;

  beforeAll(async () => {
    // Authenticate and get token
    const response = await AuthenticationService.scpApiSessionControllerCreate({
      requestBody: {
        email: process.env.ADMIN_EMAIL || 'root@scp.local',
        password: process.env.ADMIN_PASSWORD || 'Password123!',
        tenant_name: process.env.TENANT_NAME || 'scp',
      },
    }) as any;

    OpenAPI.TOKEN = response.token;
  });

  it('should create a tenant', async () => {
    const tenant = await createTenant();

    expect(tenant).toBeDefined();
    expect(tenant.id).toBeTruthy();
    expect(tenant.name).toContain('Acme Salon');
    expect(tenant.type).toBe('enterprise');

    tenantId = tenant.id;
  });

  it('should list tenants', async () => {
    const response = await listTenants();

    expect(response.data).toBeDefined();
    expect(Array.isArray(response.data)).toBe(true);
    expect(response.data!.length).toBeGreaterThan(0);
    expect(response.meta).toBeDefined();
    expect(response.meta!.total_count).toBeGreaterThan(0);
  });

  it('should get a single tenant', async () => {
    const tenant = await getTenant(tenantId);

    expect(tenant).toBeDefined();
    expect(tenant.id).toBe(tenantId);
    expect(tenant.name).toBeTruthy();
  });

  it('should update a tenant', async () => {
    const updated = await updateTenant(tenantId);

    expect(updated).toBeDefined();
    expect(updated.id).toBe(tenantId);
    expect(updated.name).toContain('Updated Salon');
  });

  it('should delete a tenant', async () => {
    await deleteTenant(tenantId);

    // Verify deletion by trying to get - should throw or return null
    await expect(async () => {
      await getTenant(tenantId);
    }).rejects.toThrow();
  });

  // Error case tests
  it('should fail to create tenant with missing required fields (422)', async () => {
    const { TenantService } = await import('@scp/sdk');

    try {
      await TenantService.tenantCreate({
        requestBody: {
          // Missing required field: name
          status: 'active',
        } as any,
      });
      expect.fail('Should have thrown validation error');
    } catch (error: any) {
      expect(error.status).toBe(422);
      expect(error.body).toBeDefined();
    }
  });

  it('should fail to get non-existent tenant (404)', async () => {
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await getTenant(fakeId);
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });

  it('should fail to update non-existent tenant (404 or 422)', async () => {
    const { TenantService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      const result = await TenantService.tenantUpdate({
        id: fakeId,
        requestBody: {
          name: 'Updated Name',
        } as any,
      });
      // If no error was thrown, check if result indicates failure
      if (result) {
        expect.fail('Should have thrown error or returned error status');
      }
    } catch (error: any) {
      // Accept error with status or any thrown error
      if (error.status) {
        expect([404, 422]).toContain(error.status);
      } else {
        // Any error is acceptable for non-existent resource
        expect(error).toBeDefined();
      }
    }
  });

  it('should fail to delete non-existent tenant (404)', async () => {
    const { TenantService } = await import('@scp/sdk');
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await TenantService.tenantDelete({ id: fakeId });
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });
});
