import { describe, it, expect, beforeAll } from 'vitest';
import { OpenAPI, AuthenticationService } from '@scp/sdk';
import { createTenant } from '@scp/client-samples/tenants/create';
import { createMerchantOrg } from '@scp/client-samples/merchant_orgs/create';
import { createMerchant } from '@scp/client-samples/merchants/create';
import { createWebsite } from '@scp/client-samples/websites/create';
import { listWebsites } from '@scp/client-samples/websites/list';
import { getWebsite } from '@scp/client-samples/websites/show';
import { updateWebsite } from '@scp/client-samples/websites/update';
import { deleteWebsite } from '@scp/client-samples/websites/delete';

describe('Website API Integration Tests', () => {
  let websiteId: string;
  let testTenantId: string;
  let testMerchantOrgId: string;
  let testMerchantId: string;

  beforeAll(async () => {
    // Authenticate
    const authResponse = (await AuthenticationService.scpApiSessionControllerCreate({
      requestBody: {
        email: process.env.ADMIN_EMAIL || 'admin@alvera-scp-dev.local',
        password: process.env.ADMIN_PASSWORD || 'DevPassword123!',
        tenant_name: process.env.TENANT_NAME || 'alvera-scp-dev',
      },
    })) as any;

    OpenAPI.TOKEN = authResponse.token;

    // Setup test data
    const tenant = await createTenant();
    testTenantId = tenant.id;

    const merchantOrg = await createMerchantOrg(testTenantId);
    testMerchantOrgId = merchantOrg.id;

    const merchant = await createMerchant(testTenantId, testMerchantOrgId);
    testMerchantId = merchant.id;
  });

  it('should create a website (201)', async () => {
    const website = await createWebsite(testTenantId, testMerchantId, testMerchantOrgId);

    expect(website).toBeDefined();
    expect(website.id).toBeTruthy();
    expect(website.domain_name).toBeTruthy();
    expect(website.title).toBeTruthy();

    websiteId = website.id;
  });

  it('should list websites (200)', async () => {
    const response = await listWebsites();

    expect(response.data).toBeDefined();
    expect(Array.isArray(response.data)).toBe(true);
  });

  it('should get a single website (200)', async () => {
    const website = await getWebsite(websiteId);

    expect(website).toBeDefined();
    expect(website.id).toBe(websiteId);
  });

  it('should update a website (200)', async () => {
    const updated = await updateWebsite(websiteId, testTenantId, testMerchantId);

    expect(updated).toBeDefined();
    expect(updated.id).toBe(websiteId);
    expect(updated.translation_allowed).toBe(true);
  });

  it('should delete a website (204)', async () => {
    await deleteWebsite(websiteId);
    // If we get here without error, the delete was successful
  });

  // Error cases
  it('should fail to create website with missing required fields (422)', async () => {
    const { WebsiteService } = await import('@scp/sdk');

    try {
      await WebsiteService.websiteCreate({
        requestBody: {
          title: 'Invalid Website',
        } as any,
      });
      expect.fail('Should have thrown validation error');
    } catch (error: any) {
      expect(error.status).toBe(422);
    }
  });

  it('should fail to get non-existent website (404)', async () => {
    const fakeId = '00000000-0000-0000-0000-000000000000';

    try {
      await getWebsite(fakeId);
      expect.fail('Should have thrown 404 error');
    } catch (error: any) {
      expect(error.status).toBe(404);
    }
  });
});
